/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*==================================================================================================
 Include Files
==================================================================================================*/
#include "EmbeddedTypes.h"
#include "tscp.h"

/*==================================================================================================
 Private macros
==================================================================================================*/
#if TSCP_USE_MEM_MANAGER
#include "MemManager.h"
#define TSCPMemAlloc(x)     MEM_BufferAlloc(x)
#define TSCPMemFree(x)      (void)MEM_BufferFree(x)
#else
#include <stdlib.h>
#define TSCPMemAlloc(x)     malloc(x)
#define TSCPMemFree(x)      free(x)
#endif  //TSCP_USE_MEM_MANAGER

#define TSCP_MAX_PARAMETER_COUNT    65535

/*==================================================================================================
 Private global variables declarations
==================================================================================================*/
static pTscpCallback_t pTscpCallbackArray[TSCP_MAX_NUMBER_OF_INSTANCES][TSCP_MAX_NB_OPCODE];
static pTscpDefaultCallback_t pTscpDefaultCallback[TSCP_MAX_NUMBER_OF_INSTANCES];

/*==================================================================================================
 Private functions
==================================================================================================*/

/*==================================================================================================
 Public functions
==================================================================================================*/
uint8_t *TSCP_PackParams (uint8_t appId, uint8_t opCode, int16_t* pParamList, uint16_t paramCount, uint16_t* pOutputBufferSize)
{
    uint16_t payloadIndex;
    tscpPacketStructureHeader_t *header;
    uint8_t* pParamListByte;
    uint8_t* packedArray;

    *pOutputBufferSize = ((paramCount * 2u) + TSCP_HEADER_OFFSET);
    packedArray = TSCPMemAlloc(*pOutputBufferSize);

    if(packedArray != NULL){
        /* Start packing the message */
        header = (tscpPacketStructureHeader_t *)(void *)packedArray;
        header->startHeader = TSCP_START_CODE_PARAMS;
        header->appId = appId;
        header->opCode = opCode;
        header->lenght = paramCount;
        pParamListByte = (uint8_t*)pParamList;
        for (payloadIndex = 0u; payloadIndex < (paramCount * 2u); payloadIndex++){
            packedArray[TSCP_HEADER_OFFSET + payloadIndex] = pParamListByte[payloadIndex];
        }
    }

    return packedArray;
}

uint8_t *TSCP_PackParamsList(uint8_t appId, uint8_t opCode, uint8_t nbListEntries, tscpParamList_t *pParamListList, uint16_t* pOutputBufferSize)
{
    uint16_t payloadIndex, paramCount;
    tscpPacketStructureHeader_t *header;
    uint8_t* pParamListByte;
    uint8_t* packedArray;
    uint8_t i, j;

    paramCount = 0u;
    for (i = 0u; i < nbListEntries; i++)
    {
        paramCount += pParamListList[i].paramCount;
    }

    *pOutputBufferSize = ((paramCount * 2u) + TSCP_HEADER_OFFSET);
    packedArray = TSCPMemAlloc(*pOutputBufferSize);

    if (packedArray != NULL)
    {
        /* Start packing the message */
        header = (tscpPacketStructureHeader_t *)(void *)packedArray;
        header->startHeader = TSCP_START_CODE_PARAMS;
        header->appId = appId;
        header->opCode = opCode;
        header->lenght = paramCount;
        payloadIndex = TSCP_HEADER_OFFSET;
        
        for (i = 0u; i < nbListEntries; i++)
        {
            pParamListByte = (uint8_t*)pParamListList[i].pParamList;
            for (j = 0u; j < (pParamListList[i].paramCount*2u); j++)
            {
                packedArray[payloadIndex] = pParamListByte[j];
                payloadIndex++;
            }
        }
    }

    return packedArray;
}   
      
void TSCP_PackDestroy (uint8_t* pPacket){
    if(pPacket != NULL){
        TSCPMemFree(pPacket);
    }
}

void TSCP_Parse (uint8_t instance, uint8_t *pPacket){
    tscpPacketStructureHeader_t* pMsgHeader = (tscpPacketStructureHeader_t*)(void *)pPacket;

    if ((pMsgHeader->startHeader == TSCP_START_CODE_PARAMS) || (pMsgHeader->startHeader == TSCP_START_CODE_RAW)){
        //This is a valid message.
        if (pMsgHeader->opCode < TSCP_MAX_NB_OPCODE){
            //The message has a valid OpCode.
            if (pTscpCallbackArray[instance][pMsgHeader->opCode] != NULL)
            {
                //A valid callback is registered for this OpCode. Execute
                pTscpCallbackArray[instance][pMsgHeader->opCode]((int16_t *)(void *)(pPacket + TSCP_HEADER_OFFSET), pMsgHeader->lenght);
            }
            else
            {
                if (pTscpDefaultCallback[instance] != NULL)
                {
                    //Execute default callback
                    pTscpDefaultCallback[instance](pPacket);
                }
            }
        }
    }
}

uint8_t TSCP_RegisterCallback (uint8_t instance, uint8_t opCode, pTscpCallback_t callbackFnc){
    static bool_t firstExecution = TRUE;
    uint8_t returnValue = 0u;

    if (firstExecution){
        //Clear the callback array the first time this function is executed
        uint8_t instanceCount, opCodeCount;
        for(instanceCount = 0u; instanceCount < TSCP_MAX_NUMBER_OF_INSTANCES; instanceCount++){
            for(opCodeCount = 0u; opCodeCount < TSCP_MAX_NB_OPCODE; opCodeCount++){
                pTscpCallbackArray[instanceCount][opCodeCount] = NULL;
            }
        }
        firstExecution = FALSE;
    }

    //Register callback
    if ((instance < TSCP_MAX_NUMBER_OF_INSTANCES) && (opCode < TSCP_MAX_NB_OPCODE)){
        //Valid instance number and opCode number. Register callback
        pTscpCallbackArray[instance][opCode] = callbackFnc;
    }
    else{
        returnValue = 1;    //Error. Invalid instance or OpCode
    }

    return returnValue;
}

uint8_t TSCP_RegisterDefaultCallback (uint8_t instance, pTscpDefaultCallback_t callbackFnc){
    static bool_t firstExecution = TRUE;
    uint8_t returnValue = 0;

    if (firstExecution){
        //Clear the callback array the first time this function is executed
        uint8_t instanceCount;
        for(instanceCount = 0u; instanceCount < TSCP_MAX_NUMBER_OF_INSTANCES; instanceCount++){
            pTscpDefaultCallback[instanceCount] = NULL;
        }
        firstExecution = FALSE;
    }

    //Register callback
    if (instance < TSCP_MAX_NUMBER_OF_INSTANCES){
        //Valid instance number and opCode number. Register callback
        pTscpDefaultCallback[instance] = callbackFnc;
    }
    else{
        returnValue = 1;    //Error. Invalid instance or OpCode
    }

    return returnValue;
}
